import React, { useState } from 'react';
import axios from 'axios';
import styles from '../../styles/ReviewWithPhotos.module.css';

export default function ReviewWithPhotos({ productId }: { productId: string }) {
  const [reviewText, setReviewText] = useState('');
  const [rating, setRating] = useState(5);
  const [photos, setPhotos] = useState<File[]>([]);

  const handlePhotoSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    setPhotos([...photos, ...(e.target.files ? Array.from(e.target.files) : [])]);
  };

  const submitReview = async () => {
    const token = localStorage.getItem('accessToken');

    // রিভিউ সাবমিট করুন
    const reviewRes = await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/products/${productId}/reviews`,
      {
        rating,
        title: `${rating} Star Review`,
        content: reviewText
      },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    const reviewId = reviewRes.data.data.id;

    // ফটো আপলোড করুন
    for (const photo of photos) {
      const formData = new FormData();
      formData.append('photo', photo);

      await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/reviews/${reviewId}/photos`,
        formData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'multipart/form-data'
          }
        }
      );
    }

    alert('Review submitted successfully!');
  };

  return (
    <div className={styles.container}>
      <h3>Write a Review with Photos</h3>

      <div className={styles.form}>
        <label>
          Rating:
          <select value={rating} onChange={(e) => setRating(parseInt(e.target.value))}>
            {[1, 2, 3, 4, 5].map((r) => (
              <option key={r} value={r}>
                {r} {'⭐'.repeat(r)}
              </option>
            ))}
          </select>
        </label>

        <textarea
          placeholder="Write your review..."
          value={reviewText}
          onChange={(e) => setReviewText(e.target.value)}
          style={{ minHeight: '150px' }}
        />

        {/* ফটো আপলোড */}
        <div className={styles.photoUpload}>
          <label>
            Upload Photos (Optional):
            <input
              type="file"
              multiple
              accept="image/*"
              onChange={handlePhotoSelect}
            />
          </label>

          {/* প্রিভিউ */}
          <div className={styles.preview}>
            {photos.map((photo, index) => (
              <div key={index} className={styles.photoItem}>
                <img
                  src={URL.createObjectURL(photo)}
                  alt={`Preview ${index}`}
                  style={{ maxWidth: '100px', maxHeight: '100px' }}
                />
                <button onClick={() => setPhotos(photos.filter((_, i) => i !== index))}>
                  Remove
                </button>
              </div>
            ))}
          </div>
        </div>

        <button onClick={submitReview}>Submit Review</button>
      </div>

      {/* রিভিউ প্রদর্শন */}
      <ReviewDisplay productId={productId} />
    </div>
  );
}

function ReviewDisplay({ productId }: { productId: string }) {
  const [reviews, setReviews] = React.useState<any[]>([]);

  React.useEffect(() => {
    const fetchReviews = async () => {
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/products/${productId}/reviews`
      );
      setReviews(res.data.reviews);
    };
    fetchReviews();
  }, [productId]);

  return (
    <div className={styles.reviewsList}>
      <h3>Customer Reviews</h3>
      {reviews.map((review) => (
        <div key={review.id} className={styles.reviewCard}>
          <div className={styles.header}>
            <p>
              {'⭐'.repeat(review.rating)} {review.verified && <span className={styles.verified}>✓ Verified</span>}
            </p>
            <small>{new Date(review.createdAt).toLocaleDateString()}</small>
          </div>

          <h4>{review.title}</h4>
          <p>{review.content}</p>

          {/* রিভিউ ফটো */}
          {review.photos && review.photos.length > 0 && (
            <div className={styles.photos}>
              {review.photos.map((photo: any) => (
                <img key={photo.id} src={photo.photoUrl} alt="Review photo" />
              ))}
            </div>
          )}
        </div>
      ))}
    </div>
  );
}