import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../styles/ProductSearch.module.css';

export default function ProductSearch({ tenantId }: { tenantId: string }) {
  const [query, setQuery] = useState('');
  const [suggestions, setSuggestions] = useState<any[]>([]);
  const [results, setResults] = useState<any[]>([]);
  const [filters, setFilters] = useState({
    categoryId: '',
    minPrice: 0,
    maxPrice: 100000
  });
  const [categories, setCategories] = useState<any[]>([]);

  useEffect(() => {
    // ফিল্টার লোড করুন
    const fetchFilters = async () => {
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/search/filters?tenantId=${tenantId}`
      );
      setCategories(res.data.categories);
    };
    fetchFilters();
  }, [tenantId]);

  // অটোকমপ্লিট
  const handleSearchChange = async (value: string) => {
    setQuery(value);

    if (value.length >= 2) {
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/search/autocomplete?tenantId=${tenantId}&q=${value}`
      );
      setSuggestions(res.data.suggestions);
    }
  };

  // সার্চ সাবমিট
  const handleSearch = async () => {
    const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/search`, {
      params: {
        tenantId,
        q: query,
        ...filters
      }
    });
    setResults(res.data.products);
  };

  return (
    <div className={styles.container}>
      {/* সার্চ ইনপুট */}
      <div className={styles.searchBox}>
        <input
          type="text"
          placeholder="Search products..."
          value={query}
          onChange={(e) => handleSearchChange(e.target.value)}
          onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
        />
        <button onClick={handleSearch}>🔍</button>

        {/* অটোকমপ্লিট ড্রপডাউন */}
        {suggestions.length > 0 && (
          <div className={styles.suggestions}>
            {suggestions.map((item) => (
              <div
                key={item.id}
                className={styles.suggestion}
                onClick={() => {
                  setQuery(item.title);
                  handleSearch();
                }}
              >
                {item.image && <img src={item.image} alt={item.title} />}
                <div>
                  <p>{item.title}</p>
                  <span>${item.price}</span>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {/* ফিল্টার সাইডবার */}
      <div className={styles.filters}>
        <h3>Filters</h3>

        <div className={styles.filterGroup}>
          <label>Category</label>
          <select
            value={filters.categoryId}
            onChange={(e) => setFilters({ ...filters, categoryId: e.target.value })}
          >
            <option value="">All Categories</option>
            {categories.map((cat) => (
              <option key={cat.id} value={cat.id}>
                {cat.name}
              </option>
            ))}
          </select>
        </div>

        <div className={styles.filterGroup}>
          <label>
            Price Range: ${filters.minPrice} - ${filters.maxPrice}
          </label>
          <input
            type="range"
            min="0"
            max="100000"
            value={filters.maxPrice}
            onChange={(e) =>
              setFilters({ ...filters, maxPrice: parseInt(e.target.value) })
            }
          />
        </div>

        <button onClick={handleSearch}>Apply Filters</button>
      </div>

      {/* রেজাল্ট */}
      <div className={styles.results}>
        {results.length > 0 ? (
          <div className={styles.grid}>
            {results.map((product) => (
              <div key={product.id} className={styles.productCard}>
                {Array.isArray(product.images) && product.images[0] && (
                  <img src={product.images[0]} alt={product.title} />
                )}
                <h4>{product.title}</h4>
                <p>${product.price}</p>
              </div>
            ))}
          </div>
        ) : (
          <p>No products found</p>
        )}
      </div>
    </div>
  );
}