import React, { useState, useEffect } from 'react';
import { io } from 'socket.io-client';
import styles from '../../styles/LiveChat.module.css';

export default function LiveChat() {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState<any[]>([]);
  const [newMessage, setNewMessage] = useState('');
  const [socket, setSocket] = useState<any>(null);
  const [sessionId, setSessionId] = useState('');

  useEffect(() => {
    // WebSocket সংযোগ
    const newSocket = io(process.env.NEXT_PUBLIC_WS_URL);

    newSocket.on('connect', () => {
      console.log('Connected to chat');
    });

    newSocket.on('new-message', (msg: any) => {
      setMessages([...messages, msg]);
    });

    setSocket(newSocket);

    return () => newSocket.close();
  }, []);

  const startChat = async () => {
    const token = localStorage.getItem('accessToken');

    const res = await fetch(
      `${process.env.NEXT_PUBLIC_API_URL}/support/chat/start`,
      {
        method: 'POST',
        headers: { Authorization: `Bearer ${token}` }
      }
    );

    const data = await res.json();
    setSessionId(data.sessionId);
    setIsOpen(true);
  };

  const sendMessage = () => {
    if (!newMessage.trim()) return;

    socket.emit('send-message', {
      sessionId,
      message: newMessage,
      isFromCustomer: true
    });

    setMessages([
      ...messages,
      { message: newMessage, isFromCustomer: true, timestamp: new Date() }
    ]);

    setNewMessage('');
  };

  return (
    <>
      {/* Chat Button */}
      <button
        className={styles.chatButton}
        onClick={startChat}
        title="Live Chat Support"
      >
        💬
      </button>

      {/* Chat Window */}
      {isOpen && (
        <div className={styles.chatWindow}>
          <div className={styles.header}>
            <h3>Live Chat Support</h3>
            <button onClick={() => setIsOpen(false)}>✕</button>
          </div>

          {/* Messages */}
          <div className={styles.messages}>
            {messages.map((msg, index) => (
              <div
                key={index}
                className={msg.isFromCustomer ? styles.customerMsg : styles.agentMsg}
              >
                <p>{msg.message}</p>
                <small>{new Date(msg.timestamp || Date.now()).toLocaleTimeString()}</small>
              </div>
            ))}
          </div>

          {/* Input */}
          <div className={styles.input}>
            <input
              type="text"
              placeholder="Type a message..."
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && sendMessage()}
            />
            <button onClick={sendMessage}>Send</button>
          </div>
        </div>
      )}
    </>
  );
}