import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../styles/ThemeToggle.module.css';

export default function ThemeToggle() {
  const [theme, setTheme] = useState<'light' | 'dark' | 'auto'>('light');

  useEffect(() => {
    // ব্যবহারকারীর পছন্দ লোড করুন
    const loadTheme = async () => {
      const token = localStorage.getItem('accessToken');
      if (token) {
        const res = await axios.get(
          `${process.env.NEXT_PUBLIC_API_URL}/user/preferences`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setTheme(res.data.theme || 'light');
      }
    };

    loadTheme();
  }, []);

  useEffect(() => {
    // DOM এ থিম প্রয়োগ করুন
    const htmlElement = document.documentElement;

    if (theme === 'auto') {
      const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
      htmlElement.setAttribute('data-theme', prefersDark ? 'dark' : 'light');
    } else {
      htmlElement.setAttribute('data-theme', theme);
    }

    // থিম সংরক্ষণ করুন
    localStorage.setItem('theme', theme);
  }, [theme]);

  const toggleTheme = async () => {
    const themes: Array<'light' | 'dark' | 'auto'> = ['light', 'dark', 'auto'];
    const currentIndex = themes.indexOf(theme);
    const nextTheme = themes[(currentIndex + 1) % themes.length];

    setTheme(nextTheme);

    // সার্ভারে সংরক্ষণ করুন
    const token = localStorage.getItem('accessToken');
    if (token) {
      await axios.put(
        `${process.env.NEXT_PUBLIC_API_URL}/user/preferences`,
        { theme: nextTheme },
        { headers: { Authorization: `Bearer ${token}` } }
      );
    }
  };

  const themeIcons = {
    light: '☀️',
    dark: '🌙',
    auto: '🔄'
  };

  return (
    <button className={styles.themeToggle} onClick={toggleTheme} title={`Switch to ${theme} mode`}>
      {themeIcons[theme]}
    </button>
  );
}