import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../styles/WholesalerChat.module.css';

export default function WholesalerChat({ wholesalerId }: { wholesalerId: string }) {
  const [messages, setMessages] = useState<any[]>([]);
  const [newMessage, setNewMessage] = useState('');
  const [connectedStores, setConnectedStores] = useState<any[]>([]);
  const [selectedStore, setSelectedStore] = useState('');

  useEffect(() => {
    fetchConnectedStores();
    // WebSocket সংযোগ স্থাপন করুন
    const ws = new WebSocket(`${process.env.NEXT_PUBLIC_WS_URL}/wholesaler-chat/${wholesalerId}`);

    ws.onmessage = (event) => {
      const message = JSON.parse(event.data);
      setMessages([...messages, message]);
    };

    return () => ws.close();
  }, [wholesalerId]);

  const fetchConnectedStores = async () => {
    const token = localStorage.getItem('accessToken');
    const res = await axios.get(
      `${process.env.NEXT_PUBLIC_API_URL}/wholesaler/${wholesalerId}/connected-stores`,
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setConnectedStores(res.data.data);
  };

  const sendMessage = async () => {
    if (!newMessage.trim() || !selectedStore) return;

    const token = localStorage.getItem('accessToken');
    await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/wholesaler/chat/send`,
      {
        wholesalerId,
        storeId: selectedStore,
        message: newMessage
      },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setNewMessage('');
  };

  return (
    <div className={styles.container}>
      <h2>Chat with Store Admins</h2>

      {/* Store Selection */}
      <select
        value={selectedStore}
        onChange={(e) => setSelectedStore(e.target.value)}
        className={styles.storeSelect}
      >
        <option value="">Select a store</option>
        {connectedStores.map((store) => (
          <option key={store.id} value={store.id}>
            {store.name}
          </option>
        ))}
      </select>

      {/* Chat Messages */}
      <div className={styles.messages}>
        {messages.map((msg) => (
          <div
            key={msg.id}
            className={msg.sender === 'wholesaler' ? styles.outgoing : styles.incoming}
          >
            <p>{msg.message}</p>
            <span className={styles.time}>{new Date(msg.createdAt).toLocaleTimeString()}</span>
          </div>
        ))}
      </div>

      {/* Input */}
      <div className={styles.inputArea}>
        <input
          type="text"
          placeholder="Type your message..."
          value={newMessage}
          onChange={(e) => setNewMessage(e.target.value)}
          onKeyPress={(e) => e.key === 'Enter' && sendMessage()}
        />
        <button onClick={sendMessage}>Send</button>
      </div>
    </div>
  );
}