import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/UserProfile.module.css';

export default function UserProfilePage() {
  const [profile, setProfile] = useState<any>(null);
  const [activeTab, setActiveTab] = useState('overview');
  const [showAddAddress, setShowAddAddress] = useState(false);

  useEffect(() => {
    const fetchProfile = async () => {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/user/profile`,
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setProfile(res.data.user);
    };

    fetchProfile();
  }, []);

  const addAddress = async (formData: any) => {
    const token = localStorage.getItem('accessToken');

    await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/user/addresses`,
      formData,
      { headers: { Authorization: `Bearer ${token}` } }
    );

    // প্রোফাইল রিফ্রেশ করুন
  };

  if (!profile) return <div>Loading...</div>;

  return (
    <div className={styles.container}>
      <h1>My Account</h1>

      <div className={styles.tabs}>
        <button
          className={activeTab === 'overview' ? styles.active : ''}
          onClick={() => setActiveTab('overview')}
        >
          Overview
        </button>
        <button
          className={activeTab === 'addresses' ? styles.active : ''}
          onClick={() => setActiveTab('addresses')}
        >
          Addresses
        </button>
        <button
          className={activeTab === 'payments' ? styles.active : ''}
          onClick={() => setActiveTab('payments')}
        >
          Payment Methods
        </button>
        <button
          className={activeTab === 'orders' ? styles.active : ''}
          onClick={() => setActiveTab('orders')}
        >
          Order History
        </button>
      </div>

      {/* Overview */}
      {activeTab === 'overview' && (
        <div className={styles.content}>
          <h2>Account Overview</h2>
          <p>Name: {profile.name}</p>
          <p>Email: {profile.email}</p>
          <p>Phone: {profile.phone}</p>
        </div>
      )}

      {/* Addresses */}
      {activeTab === 'addresses' && (
        <div className={styles.content}>
          <h2>Shipping Addresses</h2>

          <button onClick={() => setShowAddAddress(!showAddAddress)}>
            + Add New Address
          </button>

          {showAddAddress && (
            <AddressForm onSubmit={addAddress} />
          )}

          <div className={styles.addressList}>
            {profile.addresses.map((addr: any) => (
              <div key={addr.id} className={styles.addressCard}>
                <h4>{addr.name}</h4>
                <p>{addr.address}</p>
                <p>{addr.city}, {addr.zipCode}</p>
                {addr.isDefault && <span className={styles.badge}>Default</span>}
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Payment Methods */}
      {activeTab === 'payments' && (
        <div className={styles.content}>
          <h2>Payment Methods</h2>

          <div className={styles.paymentList}>
            {profile.paymentMethods.map((method: any) => (
              <div key={method.id} className={styles.paymentCard}>
                <p>{method.type}</p>
                {method.cardLast4 && <p>•••• {method.cardLast4}</p>}
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Order History */}
      {activeTab === 'orders' && (
        <div className={styles.content}>
          <h2>Recent Orders</h2>

          <div className={styles.orderList}>
            {profile.orders.map((order: any) => (
              <div key={order.id} className={styles.orderCard}>
                <p>Order #{order.orderNumber}</p>
                <p>Status: {order.status}</p>
                <p>Total: ${order.total}</p>
                <a href={`/orders/${order.id}/tracking`}>Track Order</a>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
}

function AddressForm({ onSubmit }: { onSubmit: (data: any) => void }) {
  const [formData, setFormData] = useState({
    name: '',
    phone: '',
    address: '',
    city: '',
    zipCode: '',
    country: ''
  });

  const handleSubmit = () => {
    onSubmit(formData);
  };

  return (
    <div style={{ border: '1px solid #ddd', padding: '20px', margin: '20px 0' }}>
      <input
        placeholder="Name"
        value={formData.name}
        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
      />
      <input
        placeholder="Phone"
        value={formData.phone}
        onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
      />
      <input
        placeholder="Address"
        value={formData.address}
        onChange={(e) => setFormData({ ...formData, address: e.target.value })}
      />
      <input
        placeholder="City"
        value={formData.city}
        onChange={(e) => setFormData({ ...formData, city: e.target.value })}
      />
      <button onClick={handleSubmit}>Add Address</button>
    </div>
  );
}