import React, { useEffect, useState } from 'react';
import Editor from '@monaco-editor/react';
import axios from 'axios';
import styles from '../../styles/CodeEditor.module.css';

export default function CodeEditorPage() {
  const [code, setCode] = useState('');
  const [codeType, setCodeType] = useState<'css' | 'js' | 'html'>('css');
  const [isSaving, setIsSaving] = useState(false);
  const [message, setMessage] = useState('');

  useEffect(() => {
    const fetchCode = async () => {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/admin/code/${codeType}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setCode(res.data.code);
    };

    fetchCode();
  }, [codeType]);

  const handleSave = async () => {
    setIsSaving(true);
    const token = localStorage.getItem('accessToken');

    try {
      await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/admin/code/save`,
        {
          type: codeType,
          code,
          reason: 'Manual edit'
        },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setMessage('Code saved successfully!');
      setTimeout(() => setMessage(''), 3000);
    } catch (err) {
      setMessage('Failed to save code');
    } finally {
      setIsSaving(false);
    }
  };

  return (
    <div className={styles.container}>
      <h1>Code Editor</h1>

      {/* ট্যাব */}
      <div className={styles.tabs}>
        {(['css', 'js', 'html'] as const).map((type) => (
          <button
            key={type}
            className={codeType === type ? styles.active : ''}
            onClick={() => setCodeType(type)}
          >
            {type.toUpperCase()}
          </button>
        ))}
      </div>

      {/* এডিটর */}
      <Editor
        height="600px"
        language={codeType === 'js' ? 'javascript' : codeType}
        value={code}
        onChange={(value) => setCode(value || '')}
        theme="vs-dark"
      />

      {/* অ্যাকশন */}
      <div className={styles.actions}>
        <button onClick={handleSave} disabled={isSaving}>
          {isSaving ? 'Saving...' : 'Save Changes'}
        </button>
        <p className={styles.message}>{message}</p>
      </div>
    </div>
  );
}