import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../../styles/ThemeSelector.module.css';

export default function ThemesPage() {
  const [themes, setThemes] = useState<any[]>([]);
  const [selectedTheme, setSelectedTheme] = useState('');
  const [category, setCategory] = useState('all');
  const [customizing, setCustomizing] = useState(false);

  const categories = [
    'all',
    'modern',
    'luxury',
    'ecommerce',
    'colorful',
    'minimal',
    'sport',
    'food',
    'beauty',
    'travel',
    'realestate',
    'education'
  ];

  useEffect(() => {
    const fetchThemes = async () => {
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/themes/all`
      );
      setThemes(res.data);
    };
    fetchThemes();
  }, []);

  const filteredThemes =
    category === 'all' ? themes : themes.filter((t) => t.category === category);

  const applyTheme = async (themeId: string) => {
    const token = localStorage.getItem('accessToken');

    await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/themes/apply`,
      { themeId },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setSelectedTheme(themeId);
    alert('Theme applied successfully!');
  };

  return (
    <div className={styles.container}>
      <h1>🎨 Choose Your Store Theme</h1>

      {/* ক্যাটেগরি ফিল্টার */}
      <div className={styles.categoryFilter}>
        {categories.map((cat) => (
          <button
            key={cat}
            className={category === cat ? styles.active : ''}
            onClick={() => setCategory(cat)}
          >
            {cat.charAt(0).toUpperCase() + cat.slice(1)}
          </button>
        ))}
      </div>

      {/* থিম গ্রিড */}
      <div className={styles.themeGrid}>
        {filteredThemes.map((theme) => (
          <div
            key={theme.id}
            className={`${styles.themeCard} ${
              selectedTheme === theme.id ? styles.selected : ''
            }`}
          >
            {/* প্রিভিউ */}
            <div className={styles.preview}>
              <div
                className={styles.colorPalette}
                style={{
                  background: `linear-gradient(135deg, 
                    ${theme.colors.primary} 0%, 
                    ${theme.colors.accent} 50%, 
                    ${theme.colors.secondary} 100%)`
                }}
              />
              <div className={styles.colorSwatches}>
                <div
                  className={styles.swatch}
                  style={{ background: theme.colors.primary }}
                  title="Primary"
                />
                <div
                  className={styles.swatch}
                  style={{ background: theme.colors.accent }}
                  title="Accent"
                />
                <div
                  className={styles.swatch}
                  style={{ background: theme.colors.success }}
                  title="Success"
                />
              </div>
            </div>

            {/* থিম তথ্য */}
            <div className={styles.info}>
              <h3>{theme.name}</h3>
              <p>{theme.description}</p>
              <span className={styles.category}>{theme.category}</span>
            </div>

            {/* অ্যাকশন */}
            <div className={styles.actions}>
              <button
                className={styles.applyBtn}
                onClick={() => applyTheme(theme.id)}
              >
                {selectedTheme === theme.id ? '✓ Applied' : 'Apply Theme'}
              </button>
              <button
                className={styles.customizeBtn}
                onClick={() => setCustomizing(theme.id)}
              >
                Customize
              </button>
            </div>
          </div>
        ))}
      </div>

      {/* কাস্টমাইজেশন মডাল */}
      {customizing && (
        <ThemeCustomizer
          themeId={customizing}
          onClose={() => setCustomizing(false)}
        />
      )}
    </div>
  );
}

// থিম কাস্টমাইজার কম্পোনেন্ট
function ThemeCustomizer({ themeId, onClose }: any) {
  const [colors, setColors] = useState<any>({});

  return (
    <div className={styles.modal}>
      <div className={styles.modalContent}>
        <h2>Customize Theme</h2>

        <div className={styles.colorEditor}>
          <label>
            Primary Color:
            <input
              type="color"
              value={colors.primary || '#000000'}
              onChange={(e) =>
                setColors({ ...colors, primary: e.target.value })
              }
            />
          </label>

          <label>
            Accent Color:
            <input
              type="color"
              value={colors.accent || '#0066FF'}
              onChange={(e) =>
                setColors({ ...colors, accent: e.target.value })
              }
            />
          </label>

          {/* আরও রঙ অপশন... */}
        </div>

        <div className={styles.actions}>
          <button>Save Changes</button>
          <button onClick={onClose}>Cancel</button>
        </div>
      </div>
    </div>
  );
}