import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/Affiliate.module.css';

export default function AffiliateDashboard() {
  const [stats, setStats] = useState<any>(null);
  const [links, setLinks] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchAffiliateData();
  }, []);

  const fetchAffiliateData = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const headers = { Authorization: `Bearer ${token}` };

      const [statsRes, linksRes] = await Promise.all([
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/affiliates/dashboard`, { headers }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/affiliates/links`, { headers })
      ]);

      setStats(statsRes.data.data);
      setLinks(linksRes.data.items);
    } catch (err) {
      console.error('Failed to fetch affiliate data:', err);
    } finally {
      setLoading(false);
    }
  };

  const generateNewLink = async (productId?: string) => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/affiliates/links`,
        { productId },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setLinks([...links, res.data.data]);
    } catch (err) {
      console.error('Failed to generate link:', err);
    }
  };

  if (loading) return <div className={styles.loading}>Loading...</div>;

  return (
    <div className={styles.container}>
      <h1>Affiliate Dashboard</h1>

      <div className={styles.statsGrid}>
        <div className={styles.statCard}>
          <h3>Total Links</h3>
          <p className={styles.number}>{stats?.totalLinks || 0}</p>
        </div>
        <div className={styles.statCard}>
          <h3>Total Clicks</h3>
          <p className={styles.number}>{stats?.totalClicks || 0}</p>
        </div>
        <div className={styles.statCard}>
          <h3>Conversions</h3>
          <p className={styles.number}>{stats?.totalConversions || 0}</p>
        </div>
        <div className={styles.statCard}>
          <h3>Estimated Earnings</h3>
          <p className={styles.number}>${stats?.estimatedEarnings || 0}</p>
        </div>
      </div>

      <button className={styles.createBtn} onClick={() => generateNewLink()}>
        + Create New Link
      </button>

      <table className={styles.linksTable}>
        <thead>
          <tr>
            <th>Code</th>
            <th>Tracking URL</th>
            <th>Clicks</th>
            <th>Conversions</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          {links.map((link) => (
            <tr key={link.id}>
              <td>{link.code}</td>
              <td>
                <code>{link.trackingUrl}</code>
              </td>
              <td>{link.clicks}</td>
              <td>{link.conversions}</td>
              <td>
                <button
                  onClick={() => navigator.clipboard.writeText(link.trackingUrl)}
                  className={styles.copyBtn}
                >
                  Copy
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}