import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {
  LineChart, Line,
  BarChart, Bar,
  PieChart, Pie, Cell,
  XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer
} from 'recharts';
import styles from '../../styles/AdvancedAnalytics.module.css';

export default function AdvancedAnalyticsPage() {
  const [analytics, setAnalytics] = useState<any>(null);
  const [dateRange, setDateRange] = useState({
    startDate: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000),
    endDate: new Date()
  });

  useEffect(() => {
    fetchAnalytics();
  }, [dateRange]);

  const fetchAnalytics = async () => {
    const token = localStorage.getItem('accessToken');
    const res = await axios.get(
      `${process.env.NEXT_PUBLIC_API_URL}/analytics/advanced`,
      {
        params: dateRange,
        headers: { Authorization: `Bearer ${token}` }
      }
    );

    setAnalytics(res.data.data);
  };

  if (!analytics) return <div>Loading...</div>;

  const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042'];

  return (
    <div className={styles.container}>
      <h1>Advanced Analytics</h1>

      {/* Date Range */}
      <div className={styles.dateRange}>
        <input
          type="date"
          value={dateRange.startDate.toISOString().split('T')[0]}
          onChange={(e) => setDateRange({
            ...dateRange,
            startDate: new Date(e.target.value)
          })}
        />
        <span>to</span>
        <input
          type="date"
          value={dateRange.endDate.toISOString().split('T')[0]}
          onChange={(e) => setDateRange({
            ...dateRange,
            endDate: new Date(e.target.value)
          })}
        />
      </div>

      {/* KPI Cards */}
      <div className={styles.kpis}>
        <div className={styles.kpi}>
          <h4>Total Revenue</h4>
          <p>${analytics.totalRevenue.toFixed(2)}</p>
        </div>
        <div className={styles.kpi}>
          <h4>Total Orders</h4>
          <p>{analytics.totalOrders}</p>
        </div>
        <div className={styles.kpi}>
          <h4>Avg Order Value</h4>
          <p>${analytics.avgOrderValue.toFixed(2)}</p>
        </div>
        <div className={styles.kpi}>
          <h4>New Customers</h4>
          <p>{analytics.newCustomers}</p>
        </div>
      </div>

      {/* Charts */}
      <div className={styles.chartGrid}>
        {/* Daily Revenue */}
        <div className={styles.chart}>
          <h3>Daily Revenue</h3>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={analytics.dailyRevenue}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="date" />
              <YAxis />
              <Tooltip />
              <Line type="monotone" dataKey="revenue" stroke="#0088FE" />
            </LineChart>
          </ResponsiveContainer>
        </div>

        {/* Top Products */}
        <div className={styles.chart}>
          <h3>Top Products</h3>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={analytics.topProducts}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis />
              <YAxis />
              <Tooltip />
              <Bar dataKey="revenue" fill="#00C49F" />
            </BarChart>
          </ResponsiveContainer>
        </div>

        {/* Hourly Orders */}
        <div className={styles.chart}>
          <h3>Orders by Hour</h3>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={analytics.hourlyOrders}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="hour" />
              <YAxis />
              <Tooltip />
              <Bar dataKey="count" fill="#FFBB28" />
            </BarChart>
          </ResponsiveContainer>
        </div>
      </div>
    </div>
  );
}