import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import axios from 'axios';
import styles from '../styles/Checkout.module.css';

interface CartItem {
  productId: string;
  quantity: number;
  variantData?: any;
}

export default function CheckoutPage() {
  const router = useRouter();
  const [cart, setCart] = useState<CartItem[]>([]);
  const [cartSummary, setCartSummary] = useState<any>(null);
  const [shippingAddress, setShippingAddress] = useState({
    name: '',
    street: '',
    city: '',
    state: '',
    postalCode: '',
    country: 'US',
    phone: ''
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  useEffect(() => {
    // Load cart from localStorage
    const savedCart = localStorage.getItem('cart');
    if (savedCart) {
      const items = JSON.parse(savedCart);
      setCart(items);
      validateCart(items);
    }
  }, []);

  const validateCart = async (items: CartItem[]) => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/cart/validate`,
        { items },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setCartSummary(res.data.data);
    } catch (err: any) {
      setError(err.response?.data?.message || 'Cart validation failed');
    }
  };

  const handlePlaceOrder = async () => {
    setLoading(true);
    setError('');

    try {
      const token = localStorage.getItem('accessToken');

      // Create order
      const orderRes = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/orders`,
        {
          items: cart,
          shippingAddress,
          billingAddress: shippingAddress
        },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      const orderId = orderRes.data.data.id;

      // Create payment session (use Stripe or local gateway)
      const paymentRes = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/payments/stripe/session`,
        { orderId },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      // Redirect to Stripe
      window.location.href = paymentRes.data.data.checkoutUrl;
    } catch (err: any) {
      setError(err.response?.data?.message || 'Order creation failed');
    } finally {
      setLoading(false);
    }
  };

  if (!cartSummary) return <div className={styles.loading}>Loading...</div>;

  return (
    <div className={styles.container}>
      <div className={styles.content}>
        <h1>Checkout</h1>

        {error && <p className={styles.error}>{error}</p>}

        <section className={styles.shippingSection}>
          <h2>Shipping Address</h2>
          <input
            type="text"
            placeholder="Full Name"
            value={shippingAddress.name}
            onChange={(e) => setShippingAddress({ ...shippingAddress, name: e.target.value })}
          />
          <input
            type="text"
            placeholder="Street"
            value={shippingAddress.street}
            onChange={(e) => setShippingAddress({ ...shippingAddress, street: e.target.value })}
          />
          <input
            type="text"
            placeholder="City"
            value={shippingAddress.city}
            onChange={(e) => setShippingAddress({ ...shippingAddress, city: e.target.value })}
          />
          <input
            type="text"
            placeholder="Postal Code"
            value={shippingAddress.postalCode}
            onChange={(e) =>
              setShippingAddress({ ...shippingAddress, postalCode: e.target.value })
            }
          />
          <input
            type="tel"
            placeholder="Phone"
            value={shippingAddress.phone}
            onChange={(e) => setShippingAddress({ ...shippingAddress, phone: e.target.value })}
          />
        </section>
      </div>

      <aside className={styles.summary}>
        <h2>Order Summary</h2>

        <div className={styles.items}>
          {cartSummary.items.map((item: any) => (
            <div key={item.id} className={styles.item}>
              <span>{item.title} × {item.quantity}</span>
              <span>${item.subtotal.toFixed(2)}</span>
            </div>
          ))}
        </div>

        <div className={styles.totals}>
          <div>
            <span>Subtotal:</span>
            <span>${cartSummary.subtotal.toFixed(2)}</span>
          </div>
          <div>
            <span>Tax:</span>
            <span>${cartSummary.tax.toFixed(2)}</span>
          </div>
          <div>
            <span>Shipping:</span>
            <span>${cartSummary.shipping.toFixed(2)}</span>
          </div>
          <div className={styles.total}>
            <span>Total:</span>
            <span>${cartSummary.total.toFixed(2)}</span>
          </div>
        </div>

        <button
          className={styles.checkoutBtn}
          onClick={handlePlaceOrder}
          disabled={loading}
        >
          {loading ? 'Processing...' : 'Proceed to Payment'}
        </button>
      </aside>
    </div>
  );
}