import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import axios from 'axios';
import styles from '../../styles/PaymentCheckout.module.css';

interface Gateway {
  name: string;
  currency: string;
  region: string;
  enabled: boolean;
}

export default function PaymentCheckout() {
  const router = useRouter();
  const { orderId } = router.query;

  const [gateways, setGateways] = useState<Gateway[]>([]);
  const [selectedGateway, setSelectedGateway] = useState<string>('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  useEffect(() => {
    if (orderId) {
      fetchGateways();
    }
  }, [orderId]);

  const fetchGateways = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/payments/gateways`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      setGateways(res.data.data);
      if (res.data.data.length > 0) {
        setSelectedGateway(res.data.data[0].name);
      }
    } catch (err: any) {
      setError('Failed to load payment methods');
    }
  };

  const handlePayment = async () => {
    if (!selectedGateway) {
      setError('Please select a payment method');
      return;
    }

    setLoading(true);
    setError('');

    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/payments/session`,
        { orderId, gatewayName: selectedGateway },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      const { checkoutUrl, redirectUrl } = res.data.data;

      // Redirect to gateway
      if (checkoutUrl) {
        window.location.href = checkoutUrl;
      } else if (redirectUrl) {
        window.location.href = redirectUrl;
      }
    } catch (err: any) {
      setError(err.response?.data?.message || 'Payment failed');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className={styles.container}>
      <h1>Select Payment Method</h1>

      {error && <p className={styles.error}>{error}</p>}

      <div className={styles.gatewaysGrid}>
        {gateways.map((gateway) => (
          <div
            key={gateway.name}
            className={`${styles.gatewayCard} ${
              selectedGateway === gateway.name ? styles.selected : ''
            }`}
            onClick={() => setSelectedGateway(gateway.name)}
          >
            <input
              type="radio"
              name="gateway"
              value={gateway.name}
              checked={selectedGateway === gateway.name}
              onChange={() => setSelectedGateway(gateway.name)}
            />
            <label>
              <h3>{gateway.name}</h3>
              <p>{gateway.currency}</p>
              <small>{gateway.region}</small>
            </label>
          </div>
        ))}
      </div>

      <button
        className={styles.payBtn}
        onClick={handlePayment}
        disabled={!selectedGateway || loading}
      >
        {loading ? 'Processing...' : 'Continue to Payment'}
      </button>
    </div>
  );
}