import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import axios from 'axios';
import styles from '../../styles/Shipping.module.css';

interface ShippingQuote {
  courierId: string;
  courierName: string;
  cost: number;
  estimatedDays: number;
  currency: string;
}

export default function ShippingPage() {
  const router = useRouter();
  const { orderId, fromCountry, toCountry, weight } = router.query;

  const [quotes, setQuotes] = useState<ShippingQuote[]>([]);
  const [selectedCourier, setSelectedCourier] = useState<string>('');
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (fromCountry && toCountry && weight) {
      fetchShippingQuotes();
    }
  }, [fromCountry, toCountry, weight]);

  const fetchShippingQuotes = async () => {
    try {
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/shipping/quotes`, {
        params: {
          fromCountry,
          toCountry,
          weight
        }
      });

      setQuotes(res.data.data);
      if (res.data.data.length > 0) {
        setSelectedCourier(res.data.data[0].courierId);
      }
    } catch (err) {
      console.error('Failed to fetch shipping quotes:', err);
    } finally {
      setLoading(false);
    }
  };

  const handleContinue = async () => {
    if (!selectedCourier) return;

    try {
      const token = localStorage.getItem('accessToken');
      await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/shipping/shipment`,
        {
          orderId,
          courierId: selectedCourier,
          weight: parseFloat(weight as string),
          pickupAddress: 'Warehouse Address', // TODO: Get from settings
          deliveryAddress: 'Customer Address' // TODO: Get from order
        },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      router.push(`/checkout/payment?orderId=${orderId}`);
    } catch (err) {
      console.error('Failed to create shipment:', err);
    }
  };

  if (loading) return <div className={styles.loading}>Loading shipping options...</div>;

  return (
    <div className={styles.container}>
      <h1>Select Shipping Method</h1>

      <div className={styles.quotesGrid}>
        {quotes.map((quote) => (
          <div
            key={quote.courierId}
            className={`${styles.quoteCard} ${
              selectedCourier === quote.courierId ? styles.selected : ''
            }`}
            onClick={() => setSelectedCourier(quote.courierId)}
          >
            <input
              type="radio"
              name="courier"
              value={quote.courierId}
              checked={selectedCourier === quote.courierId}
              onChange={() => setSelectedCourier(quote.courierId)}
            />
            <label>
              <h3>{quote.courierName}</h3>
              <p className={styles.cost}>${quote.cost.toFixed(2)}</p>
              <p className={styles.days}>Delivery in {quote.estimatedDays} days</p>
            </label>
          </div>
        ))}
      </div>

      <button className={styles.continueBtn} onClick={handleContinue}>
        Continue to Payment
      </button>
    </div>
  );
}