import React, { useState } from 'react';
import axios from 'axios';
import styles from '../../styles/AdvancedSettings.module.css';

export default function AdvancedSettings() {
  const [settings, setSettings] = useState({
    storeName: '',
    storeEmail: '',
    storePhone: '',
    smtpHost: '',
    smtpPort: 587,
    smtpUser: '',
    smtpPass: '',
    maintenanceMode: false,
    maintenanceText: '',
    currency: 'USD',
    taxRate: 10,
    shippingCost: 10,
    shippingFreeAfter: 100
  });

  const [bulkOps, setBulkOps] = useState({
    productIds: [],
    action: 'update_price',
    value: 0
  });

  const [loading, setLoading] = useState(false);

  const handleSettingsChange = (e: any) => {
    const { name, value, type, checked } = e.target;
    setSettings({
      ...settings,
      [name]: type === 'checkbox' ? checked : value
    });
  };

  const saveSettings = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('accessToken');
      await axios.put(
        `${process.env.NEXT_PUBLIC_API_URL}/settings`,
        settings,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      alert('Settings saved successfully');
    } catch (err) {
      alert('Failed to save settings');
    } finally {
      setLoading(false);
    }
  };

  const handleBulkOperation = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('accessToken');
      await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/products/bulk-action`,
        bulkOps,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      alert('Bulk operation completed');
    } catch (err) {
      alert('Bulk operation failed');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className={styles.container}>
      <h1>Advanced Settings</h1>

      <div className={styles.tabs}>
        <Tab label="Store Settings" active>
          <div className={styles.section}>
            <h2>Store Information</h2>
            <input
              type="text"
              name="storeName"
              placeholder="Store Name"
              value={settings.storeName}
              onChange={handleSettingsChange}
            />
            <input
              type="email"
              name="storeEmail"
              placeholder="Store Email"
              value={settings.storeEmail}
              onChange={handleSettingsChange}
            />
            <input
              type="tel"
              name="storePhone"
              placeholder="Store Phone"
              value={settings.storePhone}
              onChange={handleSettingsChange}
            />
          </div>

          <div className={styles.section}>
            <h2>SMTP Configuration</h2>
            <input
              type="text"
              name="smtpHost"
              placeholder="SMTP Host"
              value={settings.smtpHost}
              onChange={handleSettingsChange}
            />
            <input
              type="number"
              name="smtpPort"
              placeholder="SMTP Port"
              value={settings.smtpPort}
              onChange={handleSettingsChange}
            />
            <input
              type="email"
              name="smtpUser"
              placeholder="SMTP User"
              value={settings.smtpUser}
              onChange={handleSettingsChange}
            />
            <input
              type="password"
              name="smtpPass"
              placeholder="SMTP Password"
              value={settings.smtpPass}
              onChange={handleSettingsChange}
            />
          </div>

          <div className={styles.section}>
            <h2>Store Policy</h2>
            <label>
              <input
                type="checkbox"
                name="maintenanceMode"
                checked={settings.maintenanceMode}
                onChange={handleSettingsChange}
              />
              Maintenance Mode
            </label>
            <textarea
              name="maintenanceText"
              placeholder="Maintenance message..."
              value={settings.maintenanceText}
              onChange={handleSettingsChange}
            />
          </div>

          <div className={styles.section}>
            <h2>Pricing & Shipping</h2>
            <input
              type="text"
              name="currency"
              placeholder="Currency (USD, BDT, etc.)"
              value={settings.currency}
              onChange={handleSettingsChange}
            />
            <input
              type="number"
              name="taxRate"
              placeholder="Tax Rate (%)"
              value={settings.taxRate}
              onChange={handleSettingsChange}
            />
            <input
              type="number"
              name="shippingCost"
              placeholder="Base Shipping Cost"
              value={settings.shippingCost}
              onChange={handleSettingsChange}
            />
            <input
              type="number"
              name="shippingFreeAfter"
              placeholder="Free Shipping After ($)"
              value={settings.shippingFreeAfter}
              onChange={handleSettingsChange}
            />
          </div>

          <button className={styles.btn} onClick={saveSettings} disabled={loading}>
            {loading ? 'Saving...' : 'Save Settings'}
          </button>
        </Tab>

        <Tab label="Bulk Operations">
          <div className={styles.section}>
            <h2>Bulk Product Operations</h2>
            <select
              value={bulkOps.action}
              onChange={(e) => setBulkOps({ ...bulkOps, action: e.target.value })}
            >
              <option value="update_price">Update Price</option>
              <option value="update_stock">Update Stock</option>
              <option value="change_category">Change Category</option>
              <option value="add_tag">Add Tag</option>
              <option value="publish">Publish All</option>
              <option value="unpublish">Unpublish All</option>
              <option value="delete">Delete Selected</option>
            </select>

            <textarea
              placeholder="Product IDs (comma-separated) or paste CSV"
              onChange={(e) => setBulkOps({ ...bulkOps, productIds: e.target.value.split(',') })}
            />

            {['update_price', 'update_stock'].includes(bulkOps.action) && (
              <input
                type="number"
                placeholder="Value"
                value={bulkOps.value}
                onChange={(e) => setBulkOps({ ...bulkOps, value: parseFloat(e.target.value) })}
              />
            )}

            <button className={styles.btn} onClick={handleBulkOperation} disabled={loading}>
              {loading ? 'Processing...' : 'Execute Bulk Operation'}
            </button>
          </div>
        </Tab>

        <Tab label="Custom Reports">
          <ReportsTab />
        </Tab>

        <Tab label="Integrations">
          <IntegrationTab />
        </Tab>
      </div>
    </div>
  );
}

function Tab({ label, active, children }: any) {
  const [activeTab, setActiveTab] = React.useState(active);

  return (
    <div>
      <button onClick={() => setActiveTab(!activeTab)}>{label}</button>
      {activeTab && <div>{children}</div>}
    </div>
  );
}

function ReportsTab() {
  const [reportType, setReportType] = React.useState('sales');

  return (
    <div style={{ padding: '2rem' }}>
      <h2>Custom Reports</h2>
      <select value={reportType} onChange={(e) => setReportType(e.target.value)}>
        <option value="sales">Sales Report</option>
        <option value="products">Product Performance</option>
        <option value="customers">Customer Analysis</option>
        <option value="inventory">Inventory Report</option>
        <option value="affiliate">Affiliate Performance</option>
      </select>
      <button style={{ marginLeft: '1rem' }}>Generate Report</button>
    </div>
  );
}

function IntegrationTab() {
  return (
    <div style={{ padding: '2rem' }}>
      <h2>Integration Settings</h2>
      <div>
        <h3>Connected Services</h3>
        <ul>
          <li>✅ Stripe Connected</li>
          <li>✅ SSLCommerz Configured</li>
          <li>❌ Sendgrid (Setup)</li>
          <li>❌ Klaviyo (Setup)</li>
        </ul>
        <button>Configure Integration</button>
      </div>
    </div>
  );
}