import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/Analytics.module.css';
import {
  LineChart, Line, BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer
} from 'recharts';

export default function AnalyticsPage() {
  const [revenue, setRevenue] = useState<any>(null);
  const [funnel, setFunnel] = useState<any>(null);
  const [ltv, setLTV] = useState<any>(null);
  const [topProducts, setTopProducts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchAnalytics();
  }, []);

  const fetchAnalytics = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const headers = { Authorization: `Bearer ${token}` };

      const [revRes, funnelRes, ltvRes, productsRes] = await Promise.all([
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/analytics/revenue`, { headers }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/analytics/funnel`, { headers }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/analytics/ltv`, { headers }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/analytics/top-products`, { headers })
      ]);

      setRevenue(revRes.data.data);
      setFunnel(funnelRes.data.data);
      setLTV(ltvRes.data.data);
      setTopProducts(productsRes.data.data);
    } catch (err) {
      console.error('Failed to fetch analytics:', err);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <div className={styles.loading}>Loading...</div>;

  const revenueData = revenue?.revenue
    ? Object.entries(revenue.revenue).map(([key, value]) => ({
        date: key,
        revenue: value
      }))
    : [];

  return (
    <div className={styles.container}>
      <h1>Analytics Dashboard</h1>

      <div className={styles.metricsGrid}>
        <div className={styles.metricCard}>
          <h3>Total Revenue</h3>
          <p className={styles.metric}>${revenue?.total?.toFixed(2)}</p>
        </div>
        <div className={styles.metricCard}>
          <h3>Avg Order Value</h3>
          <p className={styles.metric}>$150.25</p>
        </div>
        <div className={styles.metricCard}>
          <h3>Conversion Rate</h3>
          <p className={styles.metric}>2.5%</p>
        </div>
        <div className={styles.metricCard}>
          <h3>Avg Customer LTV</h3>
          <p className={styles.metric}>${ltv?.avgLTV}</p>
        </div>
      </div>

      <div className={styles.chartsGrid}>
        <div className={styles.chartCard}>
          <h2>Revenue Trend</h2>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={revenueData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="date" />
              <YAxis />
              <Tooltip />
              <Line type="monotone" dataKey="revenue" stroke="#007bff" />
            </LineChart>
          </ResponsiveContainer>
        </div>

        <div className={styles.chartCard}>
          <h2>Conversion Funnel</h2>
          <div className={styles.funnelChart}>
            <div className={styles.funnelStep} style={{ width: '100%' }}>
              <span>Visitors: {funnel?.visitors}</span>
            </div>
            <div className={styles.funnelStep} style={{ width: funnel?.addToCart }}>
              <span>Add to Cart: {funnel?.addToCart}</span>
            </div>
            <div className={styles.funnelStep} style={{ width: funnel?.checkoutStart }}>
              <span>Checkout: {funnel?.checkoutStart}</span>
            </div>
            <div className={styles.funnelStep} style={{ width: funnel?.checkoutComplete }}>
              <span>Completed: {funnel?.checkoutComplete}</span>
            </div>
          </div>
        </div>
      </div>

      <div className={styles.tableCard}>
        <h2>Top Products</h2>
        <table>
          <thead>
            <tr>
              <th>Title</th>
              <th>Orders</th>
              <th>Revenue</th>
            </tr>
          </thead>
          <tbody>
            {topProducts.map((product) => (
              <tr key={product.id}>
                <td>{product.title}</td>
                <td>{product._count?.orders || 0}</td>
                <td>${(product.price * (product._count?.orders || 0)).toFixed(2)}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}