import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/AuditLogs.module.css';

export default function AuditLogsPage() {
  const [logs, setLogs] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filters, setFilters] = useState({
    entityType: '',
    action: '',
    startDate: ''
  });

  useEffect(() => {
    fetchLogs();
  }, [filters]);

  const fetchLogs = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/audit/logs`, {
        params: filters,
        headers: { Authorization: `Bearer ${token}` }
      });

      setLogs(res.data.items);
    } catch (err) {
      console.error('Failed to fetch logs:', err);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <div>Loading...</div>;

  return (
    <div className={styles.container}>
      <h1>System Audit Logs</h1>

      {/* ফিল্টার */}
      <div className={styles.filters}>
        <input
          type="text"
          placeholder="Entity type (Product, Order, User)"
          value={filters.entityType}
          onChange={(e) => setFilters({ ...filters, entityType: e.target.value })}
        />
        <input
          type="text"
          placeholder="Action (created, updated, deleted)"
          value={filters.action}
          onChange={(e) => setFilters({ ...filters, action: e.target.value })}
        />
        <input
          type="date"
          value={filters.startDate}
          onChange={(e) => setFilters({ ...filters, startDate: e.target.value })}
        />
      </div>

      {/* টেবিল */}
      <table className={styles.table}>
        <thead>
          <tr>
            <th>Timestamp</th>
            <th>Action</th>
            <th>Entity</th>
            <th>User</th>
            <th>IP Address</th>
            <th>Changes</th>
          </tr>
        </thead>
        <tbody>
          {logs.map((log) => (
            <tr key={log.id}>
              <td>{new Date(log.createdAt).toLocaleString()}</td>
              <td><span className={styles.badge}>{log.action}</span></td>
              <td>
                {log.entityType} #{log.entityId.slice(0, 8)}
              </td>
              <td>{log.userId?.slice(0, 8)}</td>
              <td>{log.ipAddress}</td>
              <td>
                {log.oldValues && log.newValues && (
                  <details>
                    <summary>View changes</summary>
                    <pre>{JSON.stringify(
                      Object.entries(log.newValues).reduce((acc: any, [key, val]) => {
                        if (log.oldValues[key] !== val) {
                          acc[key] = { from: log.oldValues[key], to: val };
                        }
                        return acc;
                      }, {}),
                      null,
                      2
                    )}</pre>
                  </details>
                )}
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}