import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/Billing.module.css';

const plans = [
  { id: 'STARTER', name: 'Starter', price: '$29/mo', features: ['100 products', '3 users'] },
  { id: 'PROFESSIONAL', name: 'Professional', price: '$99/mo', features: ['Unlimited', '10 users'] },
  { id: 'ENTERPRISE', name: 'Enterprise', price: 'Custom', features: ['Everything'] }
];

export default function Billing() {
  const [current, setCurrent] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    fetchSubscription();
  }, []);

  const fetchSubscription = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/billing/subscription`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setCurrent(res.data.data?.plan);
    } catch (err) {
      console.error('Failed to fetch subscription:', err);
    }
  };

  const handleUpgrade = async (planType: string) => {
    setLoading(true);
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/billing/checkout-session`,
        { planType },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      window.location.href = res.data.data.checkoutUrl;
    } catch (err) {
      console.error('Checkout failed:', err);
      setLoading(false);
    }
  };

  return (
    <div className={styles.container}>
      <h1>Billing & Plans</h1>

      <div className={styles.plansGrid}>
        {plans.map((plan) => (
          <div key={plan.id} className={`${styles.planCard} ${current === plan.id ? styles.active : ''}`}>
            <h3>{plan.name}</h3>
            <p className={styles.price}>{plan.price}</p>
            <ul>
              {plan.features.map((feature) => (
                <li key={feature}>✓ {feature}</li>
              ))}
            </ul>
            {current === plan.id ? (
              <button disabled>Current Plan</button>
            ) : (
              <button
                onClick={() => handleUpgrade(plan.id)}
                disabled={loading}
                className={styles.upgradeBtn}
              >
                {loading ? 'Processing...' : 'Upgrade'}
              </button>
            )}
          </div>
        ))}
      </div>
    </div>
  );
}