import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { LineChart, Line, BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import styles from '../../styles/Dashboard.module.css';

export default function Dashboard() {
  const [stats, setStats] = useState<any>(null);
  const [orders, setOrders] = useState<any[]>([]);
  const [topProducts, setTopProducts] = useState<any[]>([]);
  const [salesChart, setSalesChart] = useState<any[]>([]);

  useEffect(() => {
    const fetchDashboardData = async () => {
      const token = localStorage.getItem('accessToken');

      const [statsRes, ordersRes, productsRes, chartRes] = await Promise.all([
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/dashboard/stats`, {
          headers: { Authorization: `Bearer ${token}` }
        }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/orders?limit=5`, {
          headers: { Authorization: `Bearer ${token}` }
        }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/products/top?limit=5`, {
          headers: { Authorization: `Bearer ${token}` }
        }),
        axios.get(`${process.env.NEXT_PUBLIC_API_URL}/analytics/sales?days=30`, {
          headers: { Authorization: `Bearer ${token}` }
        })
      ]);

      setStats(statsRes.data);
      setOrders(ordersRes.data.items);
      setTopProducts(productsRes.data.items);
      setSalesChart(chartRes.data);
    };

    fetchDashboardData();
  }, []);

  return (
    <div className={styles.dashboard}>
      <h1>Dashboard</h1>

      {/* Key Stats */}
      <div className={styles.stats}>
        <div className={styles.card}>
          <h3>Total Revenue</h3>
          <p className={styles.value}>${stats?.totalRevenue || 0}</p>
          <p className={styles.change}>↑ 12% this month</p>
        </div>

        <div className={styles.card}>
          <h3>Total Orders</h3>
          <p className={styles.value}>{stats?.totalOrders || 0}</p>
          <p className={styles.change}>↑ 8 today</p>
        </div>

        <div className={styles.card}>
          <h3>Total Products</h3>
          <p className={styles.value}>{stats?.totalProducts || 0}</p>
          <p className={styles.change}>Active products</p>
        </div>

        <div className={styles.card}>
          <h3>Total Customers</h3>
          <p className={styles.value}>{stats?.totalCustomers || 0}</p>
          <p className={styles.change}>↑ 45 new</p>
        </div>
      </div>

      {/* Charts */}
      <div className={styles.charts}>
        <div className={styles.chartContainer}>
          <h3>Sales Over Time</h3>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={salesChart}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="date" />
              <YAxis />
              <Tooltip />
              <Legend />
              <Line type="monotone" dataKey="revenue" stroke="#007bff" />
              <Line type="monotone" dataKey="orders" stroke="#28a745" />
            </LineChart>
          </ResponsiveContainer>
        </div>

        <div className={styles.chartContainer}>
          <h3>Top Products</h3>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={topProducts}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="title" angle={-45} textAnchor="end" />
              <YAxis />
              <Tooltip />
              <Bar dataKey="sales" fill="#007bff" />
            </BarChart>
          </ResponsiveContainer>
        </div>
      </div>

      {/* Recent Orders */}
      <div className={styles.ordersSection}>
        <h3>Recent Orders</h3>
        <table className={styles.table}>
          <thead>
            <tr>
              <th>Order ID</th>
              <th>Customer</th>
              <th>Amount</th>
              <th>Status</th>
              <th>Date</th>
            </tr>
          </thead>
          <tbody>
            {orders.map((order) => (
              <tr key={order.id}>
                <td>{order.orderNumber}</td>
                <td>{order.user.name}</td>
                <td>${order.total}</td>
                <td><span className={styles[order.status]}>{order.status}</span></td>
                <td>{new Date(order.createdAt).toLocaleDateString()}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}