import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/Usage.module.css';

export default function UsagePage() {
  const [usage, setUsage] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchUsage();
  }, []);

  const fetchUsage = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/metering/usage`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      setUsage(res.data.data);
    } catch (err) {
      console.error('Failed to fetch usage:', err);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <div>Loading...</div>;
  if (!usage) return <div>No usage data</div>;

  return (
    <div className={styles.container}>
      <h1>Usage & Billing</h1>

      <div className={styles.metricsGrid}>
        {/* অর্ডার */}
        <div className={styles.metricCard}>
          <h3>Orders</h3>
          <p className={styles.usage}>
            {usage.orders?.usage} / {usage.orders?.limit}
          </p>
          <div className={styles.progressBar}>
            <div
              className={styles.progress}
              style={{
                width: `${(usage.orders?.usage / usage.orders?.limit) * 100}%`,
                backgroundColor: usage.orders?.overage > 0 ? '#ff6b6b' : '#51cf66'
              }}
            />
          </div>
          {usage.orders?.overage > 0 && (
            <p className={styles.overage}>
              Overage: {usage.orders.overage} orders (${usage.orders.cost.toFixed(2)})
            </p>
          )}
        </div>

        {/* স্টোরেজ */}
        <div className={styles.metricCard}>
          <h3>Storage</h3>
          <p className={styles.usage}>
            {(usage.storage?.usage / 1024).toFixed(1)} GB / {(usage.storage?.limit / 1024).toFixed(1)} GB
          </p>
          <div className={styles.progressBar}>
            <div
              className={styles.progress}
              style={{
                width: `${(usage.storage?.usage / usage.storage?.limit) * 100}%`,
                backgroundColor: usage.storage?.overage > 0 ? '#ff6b6b' : '#51cf66'
              }}
            />
          </div>
          {usage.storage?.overage > 0 && (
            <p className={styles.overage}>
              Overage: {(usage.storage.overage / 1024).toFixed(1)} GB (${usage.storage.cost.toFixed(2)})
            </p>
          )}
        </div>

        {/* API কল */}
        <div className={styles.metricCard}>
          <h3>API Calls</h3>
          <p className={styles.usage}>
            {usage.apiCalls?.usage.toLocaleString()} / {usage.apiCalls?.limit.toLocaleString()}
          </p>
          <div className={styles.progressBar}>
            <div
              className={styles.progress}
              style={{
                width: `${(usage.apiCalls?.usage / usage.apiCalls?.limit) * 100}%`,
                backgroundColor: usage.apiCalls?.overage > 0 ? '#ff6b6b' : '#51cf66'
              }}
            />
          </div>
          {usage.apiCalls?.overage > 0 && (
            <p className={styles.overage}>
              Overage: {usage.apiCalls.overage} calls (${usage.apiCalls.cost.toFixed(2)})
            </p>
          )}
        </div>

        {/* মোট Overage খরচ */}
        <div className={styles.metricCard}>
          <h3>Total Overage Cost</h3>
          <p className={styles.totalCost}>
            ${usage.totalOverageCost?.toFixed(2)}
          </p>
          <p className={styles.info}>
            This will be added to your next invoice
          </p>
        </div>
      </div>

      {/* চালান */}
      <section className={styles.invoiceSection}>
        <h2>Current Month Invoice Preview</h2>
        <div className={styles.invoice}>
          <p><strong>Period:</strong> {usage.period}</p>
          <p><strong>Base Plan Cost:</strong> $29.00 (Starter)</p>
          <p><strong>Order Overage:</strong> ${usage.orders?.cost.toFixed(2)}</p>
          <p><strong>Storage Overage:</strong> ${usage.storage?.cost.toFixed(2)}</p>
          <p><strong>API Overage:</strong> ${usage.apiCalls?.cost.toFixed(2)}</p>
          <hr />
          <p><strong>Subtotal:</strong> ${(29 + usage.totalOverageCost).toFixed(2)}</p>
          <p><strong>Tax (10%):</strong> ${((29 + usage.totalOverageCost) * 0.1).toFixed(2)}</p>
          <p className={styles.total}>
            <strong>Total:</strong> ${((29 + usage.totalOverageCost) * 1.1).toFixed(2)}
          </p>
        </div>
      </section>
    </div>
  );
}