import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import axios from 'axios';
import styles from '../../styles/Tracking.module.css';

interface TrackingEvent {
  status: string;
  location: string;
  message: string;
  timestamp: string;
}

interface TrackingInfo {
  trackingNumber: string;
  status: string;
  currentLocation: string;
  estimatedDelivery: string;
  events: TrackingEvent[];
}

export default function TrackingPage() {
  const router = useRouter();
  const { trackingNumber } = router.query;

  const [tracking, setTracking] = useState<TrackingInfo | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (trackingNumber) {
      fetchTracking();
    }
  }, [trackingNumber]);

  const fetchTracking = async () => {
    try {
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/shipping/track/${trackingNumber}`
      );

      setTracking(res.data.data);
    } catch (err) {
      console.error('Failed to fetch tracking:', err);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <div className={styles.loading}>Loading tracking info...</div>;
  if (!tracking) return <div className={styles.error}>Shipment not found</div>;

  const statusColors: any = {
    PENDING: '#FFA500',
    PICKED_UP: '#1E90FF',
    IN_TRANSIT: '#FF6347',
    OUT_FOR_DELIVERY: '#FFD700',
    DELIVERED: '#32CD32',
    FAILED: '#DC143C'
  };

  return (
    <div className={styles.container}>
      <h1>Order Tracking</h1>

      <div className={styles.header}>
        <div>
          <p className={styles.label}>Tracking Number</p>
          <p className={styles.value}>{tracking.trackingNumber}</p>
        </div>
        <div>
          <p className={styles.label}>Status</p>
          <p className={styles.status} style={{ color: statusColors[tracking.status] }}>
            {tracking.status}
          </p>
        </div>
        <div>
          <p className={styles.label}>Location</p>
          <p className={styles.value}>{tracking.currentLocation}</p>
        </div>
      </div>

      {tracking.estimatedDelivery && (
        <div className={styles.estimatedDelivery}>
          <p>Estimated Delivery: <strong>{new Date(tracking.estimatedDelivery).toLocaleDateString()}</strong></p>
        </div>
      )}

      <div className={styles.timeline}>
        <h2>Tracking History</h2>
        {tracking.events.map((event, idx) => (
          <div key={idx} className={styles.event}>
            <div className={styles.dot} style={{ backgroundColor: statusColors[event.status] }}></div>
            <div className={styles.content}>
              <p className={styles.eventStatus}>{event.status}</p>
              <p className={styles.eventLocation}>{event.location}</p>
              <p className={styles.eventMessage}>{event.message}</p>
              <p className={styles.eventTime}>
                {new Date(event.timestamp).toLocaleString()}
              </p>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}