import React, { useState } from 'react';
import axios from 'axios';
import styles from '../../styles/ProductImport.module.css';

export default function ProductImportPage() {
  const [file, setFile] = useState<File | null>(null);
  const [fileType, setFileType] = useState<'csv' | 'excel'>('csv');
  const [isImporting, setIsImporting] = useState(false);
  const [importStatus, setImportStatus] = useState<any>(null);

  const handleFileUpload = async () => {
    if (!file) return;

    setIsImporting(true);

    const formData = new FormData();
    formData.append('file', file);
    formData.append('fileType', fileType);

    const token = localStorage.getItem('accessToken');

    try {
      const res = await axios.post(
        `${process.env.NEXT_PUBLIC_API_URL}/products/import`,
        formData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'multipart/form-data'
          }
        }
      );

      const jobId = res.data.jobId;

      // স্ট্যাটাস পোল করুন
      const pollInterval = setInterval(async () => {
        const statusRes = await axios.get(
          `${process.env.NEXT_PUBLIC_API_URL}/products/import/${jobId}/status`,
          { headers: { Authorization: `Bearer ${token}` } }
        );

        setImportStatus(statusRes.data.data);

        if (statusRes.data.data.status === 'completed') {
          clearInterval(pollInterval);
          setIsImporting(false);
        }
      }, 2000);
    } catch (err) {
      alert('Import failed');
      setIsImporting(false);
    }
  };

  return (
    <div className={styles.container}>
      <h1>Import Products</h1>

      {!importStatus ? (
        <div className={styles.form}>
          <div className={styles.fileInput}>
            <input
              type="file"
              onChange={(e) => setFile(e.target.files?.[0] || null)}
              accept=".csv,.xlsx,.json"
            />
            {file && <p>✓ {file.name}</p>}
          </div>

          <div className={styles.fileType}>
            <label>
              <input
                type="radio"
                value="csv"
                checked={fileType === 'csv'}
                onChange={(e) => setFileType(e.target.value as 'csv')}
              />
              CSV
            </label>
            <label>
              <input
                type="radio"
                value="excel"
                checked={fileType === 'excel'}
                onChange={(e) => setFileType(e.target.value as 'excel')}
              />
              Excel (XLSX)
            </label>
          </div>

          <button
            onClick={handleFileUpload}
            disabled={!file || isImporting}
            className={styles.importBtn}
          >
            {isImporting ? 'Importing...' : 'Import Products'}
          </button>

          {/* Template Download */}
          <a href="/templates/product-import-template.csv" download className={styles.template}>
            Download Template
          </a>
        </div>
      ) : (
        <div className={styles.status}>
          <h2>Import Status</h2>
          <p>Total: {importStatus.totalRows}</p>
          <p>Success: {importStatus.successCount} ✓</p>
          <p>Failed: {importStatus.failureCount} ✗</p>

          {importStatus.errors.length > 0 && (
            <div className={styles.errors}>
              <h3>Errors:</h3>
              <ul>
                {importStatus.errors.slice(0, 10).map((err: string, i: number) => (
                  <li key={i}>{err}</li>
                ))}
              </ul>
              {importStatus.errors.length > 10 && (
                <p>+ {importStatus.errors.length - 10} more errors</p>
              )}
            </div>
          )}

          <button onClick={() => window.location.reload()}>Import Another File</button>
        </div>
      )}
    </div>
  );
}