import React, { useEffect, useState } from 'react';
import { useRouter } from 'next/router';
import axios from 'axios';
import Link from 'next/link';
import styles from '../../styles/Products.module.css';

interface Product {
  id: string;
  sku: string;
  title: string;
  price: number;
  stock: number;
  images: string[];
}

export default function ProductsPage() {
  const router = useRouter();
  const [products, setProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState('');

  useEffect(() => {
    fetchProducts();
  }, []);

  const fetchProducts = async () => {
    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/products`, {
        headers: { Authorization: `Bearer ${token}` }
      });

      setProducts(res.data.items);
    } catch (err) {
      console.error('Failed to fetch products:', err);
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = async () => {
    if (!search) return;

    try {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/products/search?q=${search}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setProducts(res.data.data);
    } catch (err) {
      console.error('Search failed:', err);
    }
  };

  if (loading) return <div className={styles.loading}>Loading...</div>;

  return (
    <div className={styles.container}>
      <h1>Products</h1>

      <div className={styles.searchBar}>
        <input
          type="text"
          placeholder="Search products..."
          value={search}
          onChange={(e) => setSearch(e.target.value)}
        />
        <button onClick={handleSearch}>Search</button>
        <Link href="/dashboard/products/new">
          <button className={styles.createBtn}>+ Add Product</button>
        </Link>
      </div>

      <div className={styles.productsGrid}>
        {products.map((product) => (
          <div key={product.id} className={styles.productCard}>
            {product.images?.[0] && (
              <img src={product.images[0]} alt={product.title} className={styles.image} />
            )}
            <h3>{product.title}</h3>
            <p className={styles.sku}>SKU: {product.sku}</p>
            <p className={styles.price}>${product.price}</p>
            <p className={product.stock > 0 ? styles.inStock : styles.outOfStock}>
              {product.stock > 0 ? `${product.stock} in stock` : 'Out of stock'}
            </p>
            <div className={styles.actions}>
              <Link href={`/products/${product.id}`}>
                <button className={styles.btn}>View</button>
              </Link>
              <Link href={`/dashboard/products/${product.id}`}>
                <button className={styles.btn}>Edit</button>
              </Link>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}