import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../styles/Integrations.module.css';

export default function IntegrationsPage() {
  const [integrations, setIntegrations] = useState<any[]>([]);
  const [showModal, setShowModal] = useState(false);
  const [selectedPlatform, setSelectedPlatform] = useState('');
  const [credentials, setCredentials] = useState<any>({});

  useEffect(() => {
    fetchIntegrations();
  }, []);

  const fetchIntegrations = async () => {
    const token = localStorage.getItem('accessToken');
    const res = await axios.get(`${process.env.NEXT_PUBLIC_API_URL}/integrations`, {
      headers: { Authorization: `Bearer ${token}` }
    });

    setIntegrations(res.data.data);
  };

  const connectPlatform = async () => {
    const token = localStorage.getItem('accessToken');

    let endpoint = '';
    let data = credentials;

    switch (selectedPlatform) {
      case 'shopify':
        endpoint = '/integrations/shopify';
        data = {
          shopifyStoreUrl: credentials.storeUrl,
          apiKey: credentials.apiKey
        };
        break;
      case 'woocommerce':
        endpoint = '/integrations/woocommerce';
        data = {
          storeUrl: credentials.storeUrl,
          consumerKey: credentials.consumerKey,
          consumerSecret: credentials.consumerSecret
        };
        break;
      case 'aliexpress':
        endpoint = '/integrations/aliexpress';
        data = { apiKey: credentials.apiKey };
        break;
    }

    try {
      await axios.post(`${process.env.NEXT_PUBLIC_API_URL}${endpoint}`, data, {
        headers: { Authorization: `Bearer ${token}` }
      });

      fetchIntegrations();
      setShowModal(false);
      setCredentials({});
    } catch (err) {
      alert('Connection failed. Please check your credentials.');
    }
  };

  return (
    <div className={styles.container}>
      <h1>Integrations</h1>

      <div className={styles.grid}>
        {/* Shopify */}
        <div className={styles.card}>
          <h3>Shopify</h3>
          <p>Sync products, orders, and inventory</p>
          {integrations.find((i) => i.platform === 'shopify')?.isActive ? (
            <div className={styles.connected}>
              <span>✓ Connected</span>
              <p>Last sync: {new Date(
                integrations.find((i) => i.platform === 'shopify')?.lastSyncedAt
              ).toLocaleDateString()}</p>
            </div>
          ) : (
            <button
              onClick={() => {
                setSelectedPlatform('shopify');
                setShowModal(true);
              }}
            >
              Connect
            </button>
          )}
        </div>

        {/* WooCommerce */}
        <div className={styles.card}>
          <h3>WooCommerce</h3>
          <p>Sync with your WordPress store</p>
          {integrations.find((i) => i.platform === 'woocommerce')?.isActive ? (
            <div className={styles.connected}>
              <span>✓ Connected</span>
            </div>
          ) : (
            <button
              onClick={() => {
                setSelectedPlatform('woocommerce');
                setShowModal(true);
              }}
            >
              Connect
            </button>
          )}
        </div>

        {/* AliExpress */}
        <div className={styles.card}>
          <h3>AliExpress</h3>
          <p>Dropshipping integration</p>
          {integrations.find((i) => i.platform === 'aliexpress')?.isActive ? (
            <div className={styles.connected}>
              <span>✓ Connected</span>
            </div>
          ) : (
            <button
              onClick={() => {
                setSelectedPlatform('aliexpress');
                setShowModal(true);
              }}
            >
              Connect
            </button>
          )}
        </div>
      </div>

      {/* Connection Modal */}
      {showModal && (
        <div className={styles.modal}>
          <div className={styles.modalContent}>
            <h2>Connect {selectedPlatform}</h2>

            {selectedPlatform === 'shopify' && (
              <>
                <input
                  type="text"
                  placeholder="Store URL (e.g., mystore.myshopify.com)"
                  value={credentials.storeUrl || ''}
                  onChange={(e) => setCredentials({ ...credentials, storeUrl: e.target.value })}
                />
                <input
                  type="text"
                  placeholder="API Key"
                  value={credentials.apiKey || ''}
                  onChange={(e) => setCredentials({ ...credentials, apiKey: e.target.value })}
                />
              </>
            )}

            {selectedPlatform === 'woocommerce' && (
              <>
                <input
                  type="text"
                  placeholder="Store URL"
                  value={credentials.storeUrl || ''}
                  onChange={(e) => setCredentials({ ...credentials, storeUrl: e.target.value })}
                />
                <input
                  type="text"
                  placeholder="Consumer Key"
                  value={credentials.consumerKey || ''}
                  onChange={(e) => setCredentials({ ...credentials, consumerKey: e.target.value })}
                />
                <input
                  type="text"
                  placeholder="Consumer Secret"
                  value={credentials.consumerSecret || ''}
                  onChange={(e) => setCredentials({ ...credentials, consumerSecret: e.target.value })}
                />
              </>
            )}

            {selectedPlatform === 'aliexpress' && (
              <input
                type="text"
                placeholder="API Key"
                value={credentials.apiKey || ''}
                onChange={(e) => setCredentials({ ...credentials, apiKey: e.target.value })}
              />
            )}

            <div className={styles.actions}>
              <button onClick={connectPlatform}>Connect</button>
              <button onClick={() => setShowModal(false)}>Cancel</button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}