import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/Performance.module.css';

export default function PerformanceMonitorPage() {
  const [performance, setPerformance] = useState<any>(null);

  useEffect(() => {
    const fetchPerformance = async () => {
      const token = localStorage.getItem('accessToken');
      const res = await axios.get(
        `${process.env.NEXT_PUBLIC_API_URL}/monitoring/performance`,
        { headers: { Authorization: `Bearer ${token}` } }
      );

      setPerformance(res.data.data);
    };

    fetchPerformance();
    const interval = setInterval(fetchPerformance, 60000); // প্রতি মিনিটে রিফ্রেশ

    return () => clearInterval(interval);
  }, []);

  if (!performance) return <div>Loading...</div>;

  const getScoreColor = (score: number) => {
    if (score >= 80) return '#28a745'; // Green
    if (score >= 60) return '#ffc107'; // Yellow
    return '#dc3545'; // Red
  };

  return (
    <div className={styles.container}>
      <h1>Store Performance</h1>

      <div className={styles.metrics}>
        {/* Page Load Time */}
        <div className={styles.card}>
          <h3>Page Load Time</h3>
          <p className={styles.value}>{(performance.avgPageLoadTime / 1000).toFixed(2)}s</p>
          <p className={styles.status}>
            {performance.avgPageLoadTime < 2000 ? '✓ Excellent' : '⚠ Needs improvement'}
          </p>
        </div>

        {/* SEO Score */}
        <div className={styles.card}>
          <h3>SEO Score</h3>
          <div className={styles.scoreBar}>
            <div
              className={styles.fill}
              style={{
                width: `${performance.seoScore}%`,
                backgroundColor: getScoreColor(performance.seoScore)
              }}
            />
          </div>
          <p className={styles.value}>{performance.seoScore}/100</p>
        </div>

        {/* Uptime */}
        <div className={styles.card}>
          <h3>Current Uptime</h3>
          <p className={styles.value}>{performance.currentUptime.toFixed(2)}%</p>
          <p className={styles.status}>30-day avg: {performance.thirtyDayUptime.toFixed(2)}%</p>
        </div>

        {/* Mobile Score */}
        <div className={styles.card}>
          <h3>Mobile Score</h3>
          <div className={styles.scoreBar}>
            <div
              className={styles.fill}
              style={{
                width: `${performance.mobileScore || 85}%`,
                backgroundColor: getScoreColor(performance.mobileScore || 85)
              }}
            />
          </div>
          <p className={styles.value}>{performance.mobileScore || 85}/100</p>
        </div>
      </div>

      {/* Recommendations */}
      <div className={styles.recommendations}>
        <h3>Improvement Recommendations</h3>
        <ul>
          {performance.avgPageLoadTime > 3000 && (
            <li>❌ Page load time is slow. Enable image compression and CDN.</li>
          )}
          {performance.seoScore < 80 && (
            <li>⚠ SEO score needs improvement. Check meta tags and structured data.</li>
          )}
          {performance.mobileScore < 80 && (
            <li>⚠ Mobile experience needs work. Test responsive design.</li>
          )}
          {performance.currentUptime < 99 && (
            <li>❌ Uptime is below 99%. Contact support for infrastructure review.</li>
          )}
        </ul>
      </div>
    </div>
  );
}