export class PlatformSDK {
  private apiKey: string;
  private apiUrl: string;
  private httpClient: any;

  constructor(apiKey: string, options?: { apiUrl?: string }) {
    this.apiKey = apiKey;
    this.apiUrl = options?.apiUrl || 'https://api.platform.com';
    this.initializeHttpClient();
  }

  private initializeHttpClient() {
    this.httpClient = axios.create({
      baseURL: this.apiUrl,
      headers: {
        'Authorization': `Bearer ${this.apiKey}`,
        'Content-Type': 'application/json'
      }
    });
  }

  // ===== PRODUCTS =====
  async getProducts(tenantId: string, options?: { limit?: number; offset?: number }) {
    return this.httpClient.get(`/tenants/${tenantId}/products`, { params: options });
  }

  async getProduct(productId: string) {
    return this.httpClient.get(`/products/${productId}`);
  }

  async createProduct(tenantId: string, data: any) {
    return this.httpClient.post(`/tenants/${tenantId}/products`, data);
  }

  async updateProduct(productId: string, data: any) {
    return this.httpClient.put(`/products/${productId}`, data);
  }

  async deleteProduct(productId: string) {
    return this.httpClient.delete(`/products/${productId}`);
  }

  // ===== ORDERS =====
  async getOrders(tenantId: string) {
    return this.httpClient.get(`/tenants/${tenantId}/orders`);
  }

  async getOrder(orderId: string) {
    return this.httpClient.get(`/orders/${orderId}`);
  }

  async createOrder(tenantId: string, data: any) {
    return this.httpClient.post(`/tenants/${tenantId}/orders`, data);
  }

  async updateOrderStatus(orderId: string, status: string) {
    return this.httpClient.patch(`/orders/${orderId}`, { status });
  }

  // ===== CUSTOMERS =====
  async getCustomers(tenantId: string) {
    return this.httpClient.get(`/tenants/${tenantId}/customers`);
  }

  async getCustomer(customerId: string) {
    return this.httpClient.get(`/customers/${customerId}`);
  }

  async createCustomer(tenantId: string, data: any) {
    return this.httpClient.post(`/tenants/${tenantId}/customers`, data);
  }

  // ===== WEBHOOKS =====
  async registerWebhook(tenantId: string, url: string, events: string[]) {
    return this.httpClient.post(`/tenants/${tenantId}/webhooks`, {
      url,
      events
    });
  }

  async getWebhooks(tenantId: string) {
    return this.httpClient.get(`/tenants/${tenantId}/webhooks`);
  }

  // ===== ANALYTICS =====
  async getAnalytics(tenantId: string, period: 'daily' | 'weekly' | 'monthly') {
    return this.httpClient.get(`/tenants/${tenantId}/analytics`, {
      params: { period }
    });
  }

  // ===== EVENTS (Real-time) =====
  onOrderCreated(callback: (order: any) => void) {
    this.subscribeTo('order.created', callback);
  }

  onProductUpdated(callback: (product: any) => void) {
    this.subscribeTo('product.updated', callback);
  }

  private subscribeTo(event: string, callback: any) {
    // WebSocket subscription logic
  }
}

// TypeScript Types
export interface Product {
  id: string;
  title: string;
  description?: string;
  price: number;
  stock: number;
  images: string[];
  categoryId?: string;
}

export interface Order {
  id: string;
  orderNumber: string;
  customerId: string;
  items: OrderItem[];
  total: number;
  status: 'pending' | 'processing' | 'shipped' | 'delivered';
}

export interface Customer {
  id: string;
  name: string;
  email: string;
  phone?: string;
  addresses: Address[];
}