import axios, { AxiosInstance } from 'axios';

export class SaaSEcommerceSDK {
  private client: AxiosInstance;
  private token?: string;

  constructor(apiUrl: string, apiKey?: string) {
    this.client = axios.create({
      baseURL: apiUrl,
      headers: {
        'Content-Type': 'application/json',
        ...(apiKey && { Authorization: `Bearer ${apiKey}` })
      }
    });
  }

  setToken(token: string) {
    this.token = token;
    this.client.defaults.headers.Authorization = `Bearer ${token}`;
  }

  // Authentication
  async register(email: string, password: string, name: string) {
    const res = await this.client.post('/auth/register', { email, password, name });
    return res.data;
  }

  async login(email: string, password: string) {
    const res = await this.client.post('/auth/login', { email, password });
    this.setToken(res.data.data.accessToken);
    return res.data;
  }

  // Products
  async createProduct(data: any) {
    const res = await this.client.post('/products', data);
    return res.data;
  }

  async listProducts(skip = 0, take = 20) {
    const res = await this.client.get('/products', { params: { skip, take } });
    return res.data;
  }

  async getProduct(id: string) {
    const res = await this.client.get(`/products/${id}`);
    return res.data;
  }

  async updateProduct(id: string, data: any) {
    const res = await this.client.put(`/products/${id}`, data);
    return res.data;
  }

  async deleteProduct(id: string) {
    const res = await this.client.delete(`/products/${id}`);
    return res.data;
  }

  // Orders
  async createOrder(data: any) {
    const res = await this.client.post('/orders', data);
    return res.data;
  }

  async listOrders(skip = 0, take = 20) {
    const res = await this.client.get('/orders', { params: { skip, take } });
    return res.data;
  }

  async getOrder(id: string) {
    const res = await this.client.get(`/orders/${id}`);
    return res.data;
  }

  // Payments
  async createPaymentSession(orderId: string, gatewayName?: string) {
    const res = await this.client.post('/payments/session', { orderId, gatewayName });
    return res.data;
  }

  // Shipping
  async getShippingQuotes(fromCountry: string, toCountry: string, weight: number) {
    const res = await this.client.get('/shipping/quotes', {
      params: { fromCountry, toCountry, weight }
    });
    return res.data;
  }

  async trackShipment(trackingNumber: string) {
    const res = await this.client.get(`/shipping/track/${trackingNumber}`);
    return res.data;
  }

  // Analytics
  async getRevenueAnalytics(period = 'daily') {
    const res = await this.client.get('/analytics/revenue', { params: { period } });
    return res.data;
  }

  async getConversionFunnel() {
    const res = await this.client.get('/analytics/funnel');
    return res.data;
  }

  // AI
  async generateProductDescription(data: any) {
    const res = await this.client.post('/ai/generate/description', data);
    return res.data;
  }

  async suggestPricing(data: any) {
    const res = await this.client.post('/ai/pricing/suggest', data);
    return res.data;
  }

  async chatbot(message: string, history?: any[]) {
    const res = await this.client.post('/ai/chatbot', { message, history });
    return res.data;
  }
}

export default SaaSEcommerceSDK;