import type { NextApiRequest, NextApiResponse } from "next";
import { validateSubdomain, normalizeSubdomain } from "../../lib/tenant";
import { prisma } from "../../lib/prisma";

type Data = {
  success: boolean;
  tenant?: {
    id: string;
    name: string;
    subdomain: string;
    host: string;
    createdAt: string;
    status?: string;
  };
  steps?: string[];
  error?: string;
};

export default async function handler(req: NextApiRequest, res: NextApiResponse<Data>) {
  if (req.method !== "POST") {
    return res.status(405).json({ success: false, error: "Method not allowed" });
  }

  const { name, subdomain } = req.body || {};

  if (!name || !subdomain) {
    return res.status(400).json({ success: false, error: "name and subdomain are required" });
  }

  const normalized = normalizeSubdomain(subdomain);
  if (!validateSubdomain(normalized)) {
    return res.status(400).json({
      success: false,
      error: "Invalid subdomain. Use a-z, 0-9 and hyphens, start/end with alphanumeric."
    });
  }

  try {
    // check existing
    const existing = await prisma.tenant.findUnique({ where: { subdomain: normalized } });
    if (existing) {
      return res.status(409).json({ success: false, error: "Subdomain already taken" });
    }

    const host = `${normalized}.${process.env.NEXT_PUBLIC_BASE_HOST || "example.com"}`;

    const tenant = await prisma.tenant.create({
      data: {
        name,
        subdomain: normalized,
        host,
        // status defaults to PROVISIONING
        metadata: {
          onboardingStep: "queued"
        }
      }
    });

    const steps = [
      "Validated input",
      "Reserved tenant record in DB",
      "Created default store settings (simulate)",
      "Queued DNS & SSL provisioning job (to be implemented)",
      "Provisioning record created"
    ];

    return res.status(201).json({
      success: true,
      tenant: {
        id: tenant.id,
        name: tenant.name,
        subdomain: tenant.subdomain,
        host: tenant.host,
        createdAt: tenant.createdAt.toISOString(),
        status: tenant.status
      },
      steps
    });
  } catch (err: any) {
    console.error("Tenant create error:", err);
    return res.status(500).json({ success: false, error: "Internal server error" });
  }
}